/*
 * Copyright (c) 1995 Eugene W. Stark
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Eugene W. Stark.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 5. No copying or redistribution in any form for commercial purposes is
 *    permitted without specific prior written permission.
 * 6. This software may be used, for evaluation purposes only, for a period
 *    of no more than fourteen days.  Use of the software for a period beyond
 *    fourteen days, or use of the software for anything other than evaluation
 *    purposes, requires written permission from the author.  Such permission
 *    may be obtained by paying to the author a registration fee,
 *    as described in the documentation accompanying this software.
 *
 * THIS SOFTWARE IS PROVIDED BY EUGENE W. STARK (THE AUTHOR) ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * HTML Output Routines
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <time.h>

#ifdef MSDOS
#include <direct.h>
#else
#include <sys/stat.h>
#endif

#ifndef HAVE_STRDUP
char *strdup(const char *s);
#endif

#include "tags.h"
#include "node.h"
#include "database.h"
#include "output.h"
#include "version.h"
#include "templates.h"
#include "interpret.h"

#ifndef FILENAME_MAX
#define FILENAME_MAX 1024
#endif

#ifdef MSDOS
char *file_template="%s.htm";
int files_per_directory = 100;
int indivs_per_directory = 1000;
int indivs_per_file = 10;
int pedigree_depth = 2;
#else
char *file_template="%s.html";
int files_per_directory = 100;
int indivs_per_directory = 100;
int indivs_per_file = 0;
int pedigree_depth = 2;
#endif

void compute_pedigree_widths(struct individual_record *rt, int depth,
			     int *widths);
void output_pedigree_info(FILE *ofile, struct individual_record *rt,
			  int maxdepth, int *widths, int curdepth,
			  unsigned int parities);
void output_pedigree_name(FILE *ofile, struct individual_record *indiv,
			  int width, char *hdr);
void pedigree_indent(FILE *ofile, int *widths, int curdepth,
		     unsigned int parities);

void output_individual(struct individual_record *rt)
{
  static FILE *ofile = NULL;
  char path[FILENAME_MAX+1];
  char url[FILENAME_MAX+1];
  char id[9];
  static int lastdirno = 0, lastfileno = 0;
  int dirno, fileno;

  if(indivs_per_directory) {
    dirno = ((rt->serial - 1) / indivs_per_directory) + 1;
    sprintf(path, "D%04d", dirno);
#ifdef MSDOS
    _mkdir(path);
#else
    mkdir(path, 0777);
#endif
    strcat(path, "/");
  } else {
    sprintf(path, "%s", "");
  }
  if(indivs_per_file) {
    fileno = ((rt->serial - 1) / indivs_per_file) + 1;
    sprintf(id, "G%07d", fileno);
    sprintf(url, file_template, id);
  } else {
    fileno = rt->serial;
    sprintf(url, file_template, rt->xref);
  }
  strcat(path, url);
  /*
   * Code assumes individuals are output in serial order
   * We attempt to speed things up by holding open a file
   * and not reopening it for each successive individual.
   * The last stream will be flushed on exit().
   */
  if(ofile == NULL || dirno != lastdirno || fileno != lastfileno) {
      if(ofile) {
	  fclose(ofile);
	  ofile = NULL;
      }
      lastdirno = dirno;
      lastfileno = fileno;
      if(indivs_per_file && rt->serial%indivs_per_file != 1) {
        ofile = fopen(path, "a");
      } else {
#ifdef MSDOS
        fprintf(stderr, "Creating %s\n", path);
#endif
        ofile = fopen(path, "w");
      }
  }
  if(ofile == NULL) {
    fprintf(stderr, "Failed to create individual file %s\n", path);
    return;
  }
  template = template_start = individual_template;
  root.indiv = rt;
  root_type = T_INDIV;
  interpret(ofile);
/*  fclose(ofile); */
}

/*
 * The source template expects the head of a list of sources as the root.
 * All source data are output to a single sources file, with bookmarks
 * for the individual sources.  This will suffice for most GEDCOMs I have
 * seen so far, in which there aren't all that many source records.
 */

void output_sources(struct source_record *srp)
{
  FILE *ofile;
  char url[FILENAME_MAX+1];

  sprintf(url, file_template, "SOURCES");
  ofile = fopen(url, "w");
  if(ofile == NULL) {
    fprintf(stderr, "Failed to create sources file %s\n", url);
    return;
  }
  template = template_start = source_template;
  root.source = srp;
  root_type = T_SOURCE;
  interpret(ofile);
  fclose(ofile);
}

/*
 * Recursive routine to create hierarchical index.
 * Assumes that inp is the root of a tree of index nodes.
 * Leaves are identifiable as those with inp->children == NULL.
 * In this case, inp->first == inp->last, a pointer to the
 * single individual at this leaf.
 * The index template expects the head of a list of index nodes as the root.
 * For each node in the list, it determines whether that node is an
 * internal node or a leaf.  If a leaf, then it outputs a leaf entry for
 * the one individual under that node.  If an internal node, it outputs
 * a range entry for that node.
 */

void output_index(struct index_node *inp)
{
  FILE *ofile;
  struct index_node *inc;
  char path[FILENAME_MAX+1];
  char base[8];

  if(inp->children == NULL)
      return;
  if(inp->id == 0)
      sprintf(path, file_template, "PERSONS");
  else {
      sprintf(base, "IND%04d", inp->id);
      sprintf(path, file_template, base);
  }
#ifdef MSDOS
    fprintf(stderr, "Creating %s\n", path);
#endif
  if((ofile = fopen(path, "w")) == NULL) {
      fprintf(stderr, "Failed to create index file %s\n", path);
      return;
  }
  template = template_start = index_template;
  root.index = inp->children;
  root_type = T_INDEX;
  doing_index = 1;
  interpret(ofile);
  doing_index = 0;
  fclose(ofile);
  for(inc = inp->children; inc != NULL; inc = inc->next)
      output_index(inc);
}

/*
 * Output surname index with links to level 1 index nodes.
 * The argument is the head of a list of index nodes, one for each surname.
 * Each index node is linked (via FIRST and LAST) to the first and last
 * individuals with that surname, and via PARENT to the level 1 index node
 * containing the first index entry for that surname.
 */

void output_surnames(struct index_node *inp)
{
  FILE *ofile;
  char path[FILENAME_MAX+1];

  sprintf(path, file_template, "SURNAMES");
#ifdef MSDOS
    fprintf(stderr, "Creating %s\n", path);
#endif
  if((ofile = fopen(path, "w")) == NULL) {
      fprintf(stderr, "Failed to create surnames file %s\n", path);
      return;
  }
  template = template_start = surname_template;
  root.index = inp;
  root_type = T_INDEX;
  doing_index = 1;
  interpret(ofile);
  doing_index = 0;
  fclose(ofile);
}

/*
 * Output index file for use by GenWeb automatic indexers.
 * Format is suitable for AWK/SORT processing:
 * each individual is on a single line, with fields terminated by '|'.
 * Lines starting with ;; are comments.
 */

void output_gendex(struct individual_record *rt)
{
  FILE *ofile;
  char path[FILENAME_MAX+1], *cp;
  struct event_structure *ep;
  int found;
  time_t tm = time(NULL);

  sprintf(path, "%s", "GENDEX.txt");
#ifdef MSDOS
    fprintf(stderr, "Creating %s\n", path);
#endif
  if((ofile = fopen(path, "w")) == NULL) {
      fprintf(stderr, "Failed to create index file %s\n", path);
      return;
  }
  /*
   * Output some version information for possible future use
   */
  fprintf(ofile, ";; %s, %s", VERSION, ctime(&tm));
  fprintf(ofile, ";; See the documentation to find out how this file can\n");
  fprintf(ofile, ";; be used to add your names to a master index.\n;;\n");
  /*
   * Now output all the individuals
   */
  for( ; rt != NULL; rt = rt->next) {
      /*
       * First field is the filename/URL
       */
      current_type = T_INDIV;
      current_value.indiv = rt;
      doing_index = 1;
      construct_url(current_url);
      doing_index = 0;
      fprintf(ofile, "%s|", current_url);
      /*
       * Second field is the surname
       */
      fprintf(ofile, "%s|", rt->personal_name && rt->personal_name->surname ?
	      rt->personal_name->surname : "");
      /*
       * Third field is the full name as from a GEDCOM file
       */
      if(rt->personal_name && rt->personal_name->name) {
	  for(cp = rt->personal_name->name; *cp != '\0'; cp++) {
	      if(cp - rt->personal_name->name
		      == rt->personal_name->surname_start
		 || cp - rt->personal_name->name
		      == rt->personal_name->surname_end)
		  fputc('/', ofile);
	      else
		  fputc(*cp, ofile);
	  }
      }
      fprintf(ofile, "|");
      /*
       * Fourth and fifth fields are the birth date and place
       */
      found = 0;
      for(ep = rt->events; ep != NULL; ep = ep->next) {
	  if(ep->tag && ep->tag->value == BIRT) {
	      found++;
	      if(ep->date)
		  fprintf(ofile, "%s", ep->date);
	      fprintf(ofile, "|");
	      if(ep->place)
		  fprintf(ofile, "%s", ep->place->name);
	      fprintf(ofile, "|");
	  }
      }
      if(found == 0)
	  fprintf(ofile, "||");
      /*
       * Sixth and seventh fields are the death date and place
       */
      found = 0;
      for(ep = rt->events; ep != NULL; ep = ep->next) {
	  if(ep->tag && ep->tag->value == DEAT) {
	      found++;
	      if(ep->date)
		  fprintf(ofile, "%s", ep->date);
	      fprintf(ofile, "|");
	      if(ep->place)
		  fprintf(ofile, "%s", ep->place->name);
	      fprintf(ofile, "|");
	  }
      }
      if(found == 0)
	  fprintf(ofile, "||");
      fprintf(ofile, "\n");
  }
  fclose(ofile);  
}

/*
 * Construction of pedigree charts
 */

#define PEDIGREE_MINW 2

void output_pedigree(FILE *ofile, struct individual_record *rt, int depth)
{
  int *widths, i;

  if((widths = malloc((depth+1) * sizeof(int))) == NULL)
    out_of_memory();
  for(i = 0; i <= depth; i++)
      widths[i] = PEDIGREE_MINW;
  compute_pedigree_widths(rt, depth, widths);
  fprintf(ofile, "<HR>\n<PRE>\n");
  output_pedigree_info(ofile, rt, depth, widths, 0, 0);
  fprintf(ofile, "</PRE>\n<HR>\n");
  free(widths);
}

void compute_pedigree_widths(struct individual_record *rt, int depth,
			     int *widths)
{
  char *cp;
  int i, space = 0;

  if(rt->personal_name) {
      for(cp = rt->personal_name->name, i = 0; *cp != '\0'; cp++) {
	  if(!space || *cp != ' ')
	      i++;
	  if(*cp == ' ')
	      space = 1;
	  else
	      space = 0;
      }
      i += 2;
      if(i > *widths)
	  *widths = i;
  }
  if(depth && rt->famc && rt->famc->pointer.family) {
    if(rt->famc->pointer.family->husband
	&& rt->famc->pointer.family->husband->pointer.individual)
      compute_pedigree_widths(rt->famc->pointer.family->
			      husband->pointer.individual,
			      depth-1, widths+1);
    if(rt->famc->pointer.family->wife
	&& rt->famc->pointer.family->wife->pointer.individual)
      compute_pedigree_widths(rt->famc->pointer.family->
			      wife->pointer.individual,
			      depth-1, widths+1);
  }
}

void output_pedigree_info(FILE *ofile, struct individual_record *rt,
			  int maxdepth, int *widths, int curdepth,
			  unsigned int parities)
{
    struct family_record *family = NULL;
    struct individual_record *husband = NULL, *wife = NULL;

    if(curdepth > maxdepth)
	return;
    if(rt && rt->famc) {
	family = rt->famc->pointer.family;
	if(family && family->husband)
	    husband = family->husband->pointer.individual;
	if(family && family->wife)
	    wife = family->wife->pointer.individual;
    }
    if(curdepth == 0) {
	output_pedigree_info(ofile, husband, maxdepth,
				 widths, 1, parities);
	fprintf(ofile, "|\n");
	fprintf(ofile, "|--");

	output_pedigree_name(ofile, rt, 0, "");
	fprintf(ofile, "\n");
	fprintf(ofile, "|\n");
	output_pedigree_info(ofile, wife, maxdepth,
			     widths, 1, parities | 1);
    } else if(curdepth == maxdepth-1 && maxdepth > 3) {
	output_pedigree_info(ofile, husband, maxdepth,
			     widths, curdepth+1, parities);
	pedigree_indent(ofile, widths, curdepth, parities);
	output_pedigree_name(ofile, rt, widths[curdepth], "_");
	fprintf(ofile, "|");
	output_pedigree_name(ofile, wife, widths[curdepth+1], "_");
	fprintf(ofile, "\n");
    } else {
	output_pedigree_info(ofile, husband, maxdepth,
			     widths, curdepth+1, parities);
	pedigree_indent(ofile, widths, curdepth, parities);
	output_pedigree_name(ofile, rt, widths[curdepth], "_");
	if(curdepth < maxdepth)
	    fprintf(ofile, "|");
	fprintf(ofile, "\n");
	output_pedigree_info(ofile, wife, maxdepth, widths,
			     curdepth+1, parities | (1 << curdepth));
    }
}

void output_pedigree_name(FILE *ofile, struct individual_record *indiv,
			  int width, char *hdr)
{
    char *np;
    int space = 0;

    if(indiv) {
	current_value.indiv = indiv;
	current_type = T_INDIV;
	construct_url(current_url);
    } else
	*current_url = '\0';
    fprintf(ofile, "<A HREF=\"%s\">", current_url);
    for(np = hdr; *np != '\0'; np++, width--)
	fputc(*np, ofile);
    if(indiv && indiv->personal_name)
	np = indiv->personal_name->name;
    else
	np = "";
    for( ; *np != '\0'; np++) {
	if(!space || *np != ' ') {
	  fputc(*np, ofile);
	  width--;
	}
	if(*np == ' ')
	    space = 1;
	else
	    space = 0;
    }
    while(width-- > 0)
	fputc('_', ofile);
    fprintf(ofile, "</A>");
}

void pedigree_indent(FILE *ofile, int *widths, int curdepth,
		     unsigned int parities)
{
    int i, j, p, q;

    for(i = 1; i < curdepth; i++) {
	/*
	 * Determine whether the line being printed is in the middle
	 * two quarters of the subtree headed at depth i.
	 */
	p = parities & (1<<(i-1)) ? (~0) : 0;
	q = parities & (1<<i) ? (~0) : 0;
	if(p ^ q)
	    fprintf(ofile, "|");
	else
	    fprintf(ofile, " ");
	for(j = 0; j < widths[i]; j++)
	    fprintf(ofile, " ");
    }
    if(parities & (1<<(curdepth-1)))
	fprintf(ofile, "|");
    else
	fprintf(ofile, " ");
}
