/*
 * Copyright (c) 1995 Eugene W. Stark
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Eugene W. Stark.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 5. No copying or redistribution in any form for commercial purposes is
 *    permitted without specific prior written permission.
 * 6. This software may be used, for evaluation purposes only, for a period
 *    of no more than fourteen days.  Use of the software for a period beyond
 *    fourteen days, or use of the software for anything other than evaluation
 *    purposes, requires written permission from the author.  Such permission
 *    may be obtained by paying to the author a registration fee,
 *    as described in the documentation accompanying this software.
 *
 * THIS SOFTWARE IS PROVIDED BY EUGENE W. STARK (THE AUTHOR) ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * Output interpreter
 */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>

#ifdef MSDOS
#include <ctype.h>
#endif

#include "tags.h"
#include "node.h"
#include "database.h"
#include "interpret.h"
#include "output.h"
#include "version.h"
#include "templates.h"

/*
 * Interpreter state
 */

int skipping;

#define CONTROL_STACK_SIZE 100
char *while_stack[CONTROL_STACK_SIZE];
int while_stack_top;
int if_stack[CONTROL_STACK_SIZE];
int if_stack_top;

char *template;
char *template_start;
union value current_value;
record_type current_type = T_INTEGER;
union value root;
record_type root_type;

char current_url[FILENAME_MAX+1];
int doing_index = 0;

void interpret(FILE *ofile);
void variable(FILE *ofile);
void command(FILE *ofile);
void collect_identifier(char **ret);
void skip_white_space();
void output_error(char *msg);

void set_variable(char *name, int value);
int get_variable(char *name);

void family_select(char *field);
void indiv_select(char *field);
void event_select(char *field);
void attr_select(char *field);
void note_select(char *field);
void source_select(char *field);
void cont_select(char *field);
void place_select(char *field);
void xref_select(char *field);
void index_select(char *field);

void construct_url(char *dest);

/*
 * Interpret the template stored in "template", outputting results
 * to "ofile".  The value "root" is the starting point for
 * the interpretation of variables.  Its type is given by "root_type".
 */

void interpret(FILE *ofile)
{
  char c;
  int start_of_line = 1;

  while((c = *template++) != '\0') {
    switch(c) {
    case '\n':
      start_of_line = 1;
      /* fall through intentional */
    case ' ':
    case '\t':
      if(!skipping)
	fputc(c, ofile);
      continue;
    case '!':
      if(!start_of_line) {
	if(!skipping)
	  fputc(c, ofile);
	continue;
      }
      template--;
      command(ofile);
      continue;
    case '$':
      start_of_line = 0;
      variable(ofile);
      if(!skipping) {
	if(current_type == T_STRING) {
	  fprintf(ofile, "%s", current_value.string);
	} else if(current_type == T_URL) {
	  fprintf(ofile, current_value.url);
	} else if(current_type == T_INTEGER) {
	  /* Integer variables start from 1 */
	  fprintf(ofile, "%d", current_value.integer + 1);
	} else {
	  output_error("Attempt to output something not an integer or string");
	}
      }
      continue;
    default:
      start_of_line = 0;
      if(!skipping)
	fputc(c, ofile);
      continue;
    }
  }
}

/*
 * After having seen the initial $, interpret a simple or compound variable. 
 */

void variable(FILE *ofile)
{
  char c, *selector;
  int braces = 0;
  int first = 1;
  int current_index;
  /*
   * $$ means output a single $
   */
  if(*template == '$') {
    if(!skipping)
      fputc(*template++, ofile);
    return;
  }
  while((c = *template) != '\0') {
    switch(c) {
      /*
       * An '@' indicates the current individual
       */
    case '@':
      first = 0;
      template++;
      if(!skipping) {
	current_value = root;
	current_type = root_type;
      }
      if(*template == '.') {
	template++;
	continue;
      } else if(*template == '[') 
	continue;
      else
	return;
      /*
       * Braces in variables simply serve as delimiters
       */
    case '{':
      template++;
      braces++;
      continue;
    case '}':
      template++;
      if(braces) {
	braces--;
	if(braces)
	  continue;
	else
	  return;
      } else {
	if(!skipping)
	  fputc(c, ofile);
	return;
      }
      /*
       * Brackets indicate integer subscripts
       */
    case '[':
      first = 0;
      template++;
      /*
       * Integer constant
       */
      if(*template >= '0' && *template <= '9') {
	/*
	 * Convert integer value and leave on top of stack
	 * (if not skipping)
	 */
      }
      /*
       * Variable subscript
       */
      else {
	record_type previous_type;
	union value previous_value;
	
	previous_type = current_type;
	previous_value = current_value;
	variable(ofile);
	if(!skipping && current_type != T_INTEGER)
	  output_error("Subscript is not an integer variable");
	else {
	  current_index = current_value.integer;
	  current_type = previous_type;
	  current_value = previous_value;
	}
	if(*template != ']') {
	  output_error("Subscript fails to end with ']'");
	} else {
	  template++;
	}
      }
      if(!skipping) {
	switch(current_type) {
	case T_INTEGER:
	case T_STRING:
	  output_error("Can't apply subscript to an integer or string");
	  break;
	case T_PLACE:
	  while(current_index--) place_select("NEXT");
	  break;
	case T_NOTE:
	  while(current_index--) note_select("NEXT");
	  break;
	case T_SOURCE:
	  while(current_index--) source_select("NEXT");
	  break;
	case T_CONT:
	  while(current_index--) cont_select("NEXT");
	  break;
	case T_EVENT:
	  while(current_index--) event_select("NEXT");
	  break;
	case T_INDIV:
	  while(current_index--) indiv_select("NEXT");
	  break;
	case T_ATTRIBUTE:
	  while(current_index--) attr_select("NEXT");
	  break;
	case T_FAMILY:
	  while(current_index--) family_select("NEXT");
	  break;
	case T_XREF:
	  while(current_index--) xref_select("NEXT");
	  break;
	case T_INDEX:
	  while(current_index--) index_select("NEXT");
	  break;
	default:
	  break;
	}
      }
      if(*template == '.') {
	template++;
	continue;
      } else if(*template == '[' || *template == '}') 
	continue;
      else
	return;
      /*
       * An ampersand means turn the current individual or index into a URL
       */
    case '&':
      template++;
      if(!skipping) {
	  construct_url(current_url);
	  current_type = T_URL;
	  current_value.url = current_url;
      }
      if(*template == '}')
	continue;
      else
	return;
      /*
       * Alphabetic characters indicate selector name.
       * Anything else is a delimiter.
       */
    default:
      collect_identifier(&selector);
      if(*selector == '\0')
	return;
      if(first) {
	if(!skipping) {
	  current_value.integer = get_variable(selector);
	  current_type = T_INTEGER;
	}
	if(*template == '}')
	  continue;
	else
	  return;
      }
      if(!skipping) {
	switch(current_type) {
	case T_INTEGER:
	case T_STRING:
	  output_error("Can't apply selector to an integer or string");
	  break;
	case T_PLACE:
	  place_select(selector);
	  break;
	case T_NOTE:
	  note_select(selector);
	  break;
	case T_SOURCE:
	  source_select(selector);
	  break;
	case T_CONT:
	  cont_select(selector);
	  break;
	case T_EVENT:
	  event_select(selector);
	  break;
	case T_INDIV:
	  indiv_select(selector);
	  break;
	case T_FAMILY:
	  family_select(selector);
	  break;
	case T_ATTRIBUTE:
	  attr_select(selector);
	  break;
	case T_XREF:
	  xref_select(selector);
	  break;
	case T_INDEX:
	  index_select(selector);
	  break;
	default:
	  break;
	}
      }
      if(*template == '.') {
	template++;
	continue;
      } else if(*template == '[' || *template == '}') 
	continue;
      else
	return;
    }
  }
}

/*
 * Record field selection operations
 */

void family_select(char *field)
{
  struct family_record *r = current_value.family;
  if(!strcmp(field, "XREF")) {
    current_type = T_STRING;
    current_value.string = (r && r->xref) ? r->xref: "";
  } else if(!strcmp(field, "REFN")) {
    current_type = T_STRING;
    current_value.string = (r && r->refn) ? r->refn: "";
  } else if(!strcmp(field, "HUSBAND")) {
    current_type = T_INDIV;
    current_value.indiv =
      (r && r->husband) ? r->husband->pointer.individual : NULL;
  } else if(!strcmp(field, "WIFE")) {
    current_type = T_INDIV;
    current_value.indiv =
      (r && r->wife) ? r->wife->pointer.individual: NULL;
  } else if(!strcmp(field, "CHILDREN")) {
    current_type = T_XREF;
    current_value.xref =
      (r && r->children) ? r->children: NULL;
  } else if(!strcmp(field, "NOTE")) {
    current_type = T_NOTE;
    current_value.note = r ? r->notes: NULL;
  } else if(!strcmp(field, "EVENT")) {
    current_type = T_EVENT;
    current_value.event = r ? r->events: NULL;
  } else if(!strcmp(field, "SOURCE")) {
    current_type = T_XREF;
    current_value.xref =
      (r && r->sources) ? r->sources: NULL;
  } else if(!strcmp(field, "NEXT")) {
    current_value.family = r ? r->next: NULL;
  } else {
    output_error("Unrecognized selector applied to family record");
  }
}

void indiv_select(char *field)
{
  struct individual_record *r = current_value.indiv;
  if(!strcmp(field, "XREF")) {
    current_type = T_STRING;
    current_value.string = (r && r->xref) ? r->xref: "";
  } else if(!strcmp(field, "NAME")) {
    current_type = T_STRING;
    current_value.string = (r && r->personal_name) ?
      r->personal_name->name: "???";
  } else if(!strcmp(field, "SURNAME")) {
    current_type = T_STRING;
    current_value.string = (r && r->personal_name
			    && r->personal_name->surname) ?
      r->personal_name->surname: "???";
  } else if(!strcmp(field, "TITLE")) {
    current_type = T_STRING;
    current_value.string = (r && r->title) ? r->title: "";
  } else if(!strcmp(field, "ISMALE")) {
    current_type = T_INTEGER;
    current_value.integer = (r && r->sex == 'M');
  } else if(!strcmp(field, "ISFEMALE")) {
    current_type = T_INTEGER;
    current_value.integer = (r && r->sex == 'F');
  } else if(!strcmp(field, "ATTR")) {
    current_type = T_ATTRIBUTE;
    current_value.attr = r ? r->attrs : NULL;
  } else if(!strcmp(field, "FAMC")) {
    current_type = T_XREF;
    current_value.xref = r ? r->famc: NULL;
  } else if(!strcmp(field, "FAMS")) {
    current_type = T_XREF;
    current_value.xref = r ? r->fams: NULL;
  } else if(!strcmp(field, "FATHER")) {
    current_value.indiv =
      (r && r->famc && r->famc->pointer.family
       && r->famc->pointer.family->husband)
	? r->famc->pointer.family->husband->pointer.individual: NULL;
  } else if(!strcmp(field, "MOTHER")) {
    current_value.indiv =
      (r && r->famc && r->famc->pointer.family
       && r->famc->pointer.family->wife)
	? r->famc->pointer.family->wife->pointer.individual: NULL;
  } else if(!strcmp(field, "NOTE")) {
    current_type = T_NOTE;
    current_value.note = r ? r->notes: NULL;
  } else if(!strcmp(field, "SOURCE")) {
    current_type = T_XREF;
    current_value.xref =
      (r && r->sources) ? r->sources: NULL;
  } else if(!strcmp(field, "EVENT")) {
    current_type = T_EVENT;
    current_value.event = r ? r->events: NULL;
  } else if(!strcmp(field, "BIRTH")) {
    struct event_structure *ep;
    current_type = T_EVENT;
    current_value.event = NULL;
    for(ep = r->events; ep != NULL; ep = ep->next) {
      if(ep->tag->value == BIRT)
	current_value.event = ep;
    }
  } else if(!strcmp(field, "DEATH")) {
    struct event_structure *ep;
    current_type = T_EVENT;
    current_value.event = NULL;
    for(ep = r->events; ep != NULL; ep = ep->next) {
      if(ep->tag->value == DEAT)
	current_value.event = ep;
    }
  } else if(!strcmp(field, "NEXT")) {
    current_value.indiv = r ? r->next: NULL;
  } else {
    output_error("Unrecognized selector applied to individual record");
  }
}

void event_select(char *field)
{
  struct event_structure *r = current_value.event;
  if(!strcmp(field, "TAG")) {
    current_type = T_STRING;
    current_value.string = (r && r->tag) ? r->tag->pname[default_language]: "";
  } else if(!strcmp(field, "DATE")) {
    current_type = T_STRING;
    current_value.string = (r && r->date) ? r->date: "";
  } else if(!strcmp(field, "PLACE")) {
    current_type = T_PLACE;
    current_value.place = r ? r->place: NULL;
  } else if(!strcmp(field, "TEXT")) {
    current_type = T_STRING;
    current_value.string = r ? r->text: "";
  } else if(!strcmp(field, "SOURCE")) {
    current_type = T_XREF;
    current_value.xref =
      (r && r->sources) ? r->sources: NULL;
  } else if(!strcmp(field, "NEXT")) {
    current_value.event = r ? r->next: NULL;
  } else {
    output_error("Unrecognized selector applied to event structure");
  }
}

void attr_select(char *field)
{
  struct attribute *r = current_value.attr;
  if(!strcmp(field, "TAG")) {
    current_type = T_STRING;
    current_value.string = (r && r->tag) ? r->tag->pname[default_language]: "";
  } else if(!strcmp(field, "TEXT")) {
    current_type = T_STRING;
    current_value.string = (r && r->text) ? r->text: "";
  } else if(!strcmp(field, "NEXT")) {
    current_value.attr = r ? r->next: NULL;
  } else {
    output_error("Unrecognized selector applied to attribute structure");
  }
}

void note_select(char *field)
{
  struct note_structure *r = current_value.note;

  if(!strcmp(field, "XREF")) {
    current_type = T_STRING;
    current_value.string = (r && r->xref) ? r->xref: "";
  } else if(!strcmp(field, "TEXT")) {
    current_type = T_STRING;
    current_value.string = (r && r->text) ? r->text: "";
  } else if(!strcmp(field, "NEXT")) {
    current_value.note = r ? r->next : NULL;
  } else if(!strcmp(field, "CONT")) {
    current_type = T_CONT;
    current_value.cont = r ? r->cont: NULL;
  } else if(!strcmp(field, "SOURCE")) {
    current_type = T_XREF;
    current_value.xref =
      (r && r->sources) ? r->sources: NULL;
  } else {
    output_error("Unrecognized selector applied to note structure");
  }
}

void source_select(char *field)
{
  struct source_record *r = current_value.source;

  if(!strcmp(field, "XREF")) {
    current_type = T_STRING;
    current_value.string = (r && r->xref) ? r->xref: "";
  } else if(!strcmp(field, "TEXT")) {
    current_type = T_STRING;
    current_value.string = (r && r->source->text) ? r->source->text: "";
  } else if(!strcmp(field, "CONT")) {
    current_type = T_CONT;
    current_value.cont = r ? r->source->cont: NULL;
  } else if(!strcmp(field, "NOTE")) {
    current_type = T_NOTE;
    current_value.note = r ? r->source->notes: NULL;
  } else if(!strcmp(field, "SOURCE")) {
    /*
     * Chained sources are not currently implemented.
     * The r->source->chain field should be an xref, but
     * we don't know whether it points to a source record
     * or to an event record.  Type information has to go
     * in the xref so that we know at this point in order
     * to put the right thing in current_type/current_value
     */
    if(r && r->source && r->source->chain) {
	if(r->source->chain->type == T_SOURCE)
	    current_value.source = r->source->chain->pointer.source;
	else {
	    current_value.source = NULL;
	    output_error("Source record chained to non-source");
	}
    } else {
	current_value.source = NULL;
    }
  } else {
    output_error("Unrecognized selector applied to source record");
  }
}

void cont_select(char *field)
{
  struct continuation *c = current_value.cont;

  if(!strcmp(field, "TEXT")) {
    current_type = T_STRING;
    current_value.string = (c && c->text) ? c->text: "";
  } else if(!strcmp(field, "BREAK")) {
    current_type = T_INTEGER;
    current_value.integer = c->linebreak;
  } else if(!strcmp(field, "NEXT")) {
    current_value.cont = c ? c->next: NULL;
  } else {
    output_error("Unrecognized selector applied to continuation structure");
  }
}

void place_select(char *field)
{
  struct place_structure *r = current_value.place;
  if(!strcmp(field, "NAME")) {
    current_type = T_STRING;
    current_value.string = (r && r->name) ? r->name: "";
  } else {
    output_error("Unrecognized selector applied to place structure");
  }
}

void xref_select(char *field)
{
  struct xref *r = current_value.xref;
  if(!strcmp(field, "INDIV")) {
    current_type = T_INDIV;
    current_value.indiv = r ? r->pointer.individual: NULL;
  } else if(!strcmp(field, "FAMILY")) {
    current_type = T_FAMILY;
    current_value.family = r ? r->pointer.family: NULL;
  } else if(!strcmp(field, "SOURCE")) {
    current_type = T_SOURCE;
    current_value.source = r ? r->pointer.source: NULL;
  } else if(!strcmp(field, "NEXT")) {
    current_value.xref = r ? r->next: NULL;
  } else {
    output_error("Unrecognized selector applied to cross-reference");
  }
}

void index_select(char *field)
{
  struct index_node *r = current_value.index;
  if(!strcmp(field, "FIRST")) {
    current_type = T_INDIV;
    current_value.indiv = r ? r->first: NULL;
  } else if(!strcmp(field, "LAST")) {
    current_type = T_INDIV;
    current_value.indiv = r ? r->last: NULL;
  } else if(!strcmp(field, "PARENT")) {
    current_value.index = r ? r->parent: NULL;
  } else if(!strcmp(field, "CHILDREN")) {
    current_value.index = r ? r->children: NULL;
  } else if(!strcmp(field, "NEXT")) {
    current_value.index = r ? r->next: NULL;
  } else if(!strcmp(field, "PREV")) {
    current_value.index = r ? r->prev: NULL;
  } else {
    output_error("Unrecognized selector applied to cross-reference");
  }
}

/*
 * Perform a control command.
 */

void command(FILE *ofile)
{
  char *buf;
  char *start = template++; 

  collect_identifier(&buf);
  skip_white_space();
  if(!strcmp(buf, "RESET")) {
    collect_identifier(&buf);
    if(*template == '\n')
      template++;
    else {
      output_error("Newline expected");
    }
    if(!skipping)
	set_variable(buf, 0);
  } else if(!strcmp(buf, "INCREMENT")) {
    collect_identifier(&buf);
    if(*template == '\n')
      template++;
    else {
      output_error("Newline expected");
    }
    if(!skipping)
	set_variable(buf, get_variable(buf)+1);
  } else if(!strcmp(buf, "IF")) {
    variable(ofile);
    if(*template == '\n') {
      template++;
      if(current_type == T_STRING) {
	if(!strcmp(current_value.string, "")) {
	  skipping++;
	  if_stack[if_stack_top++] = 1;
	} else {
	  if_stack[if_stack_top++] = 0;
	}
      } else {
	if(!current_value.integer) {
	  skipping++;
	  if_stack[if_stack_top++] = 1;
	} else {
	  if_stack[if_stack_top++] = 0;
	}
      }
    } else {
      output_error("Newline expected");
    }
  } else if(!strcmp(buf, "ELSE")) {
    if(*template == '\n') {
      template++;
      if(if_stack[if_stack_top-1]) {
	skipping--;
	if_stack[if_stack_top-1] = 0;
      } else {
	skipping++;
	if_stack[if_stack_top-1] = 1;
      }
    } else {
      output_error("Newline expected");
    }
  } else if(!strcmp(buf, "ENDIF")) {
    if(*template == '\n') {
      template++;
      if(if_stack[--if_stack_top])
	skipping--;
    } else {
      output_error("Newline expected");
    }
  } else if(!strcmp(buf, "WHILE")) {
    variable(ofile);
    if(*template == '\n') {
      template++;
      if(while_stack_top < CONTROL_STACK_SIZE-1)
	  while_stack[while_stack_top++] = start;
      else {
	  output_error("Output interpreter stack overflow");
	  exit(1);
      }
      if(!skipping) {
	if(current_type == T_STRING) {
	  if(!strcmp(current_value.string, ""))
	    skipping++;
	} else {
	  if(!current_value.integer)
	    skipping++;
	}
      } else {
	skipping++;
      }
    } else {
      output_error("Newline expected");
    }
  } else if(!strcmp(buf, "END")) {
    if(*template == '\n') {
      template++;
      if(skipping) {
	skipping--;
        if(while_stack_top)
	  while_stack_top--;
      } else {
        if(while_stack_top)
	  template = while_stack[--while_stack_top];
      }
    } else {
      output_error("Newline expected");
    }
  } else if(!strcmp(buf, "NEXT")) {
    if(*template == '\n') {
      template++;
      if(!skipping) {
	  if(root_type == T_INDIV && root.indiv)
	      root.indiv = root.indiv->next;
	  else if(root_type == T_INDEX && root.index)
	      root.index = root.index->next;
	  else if(root_type == T_SOURCE && root.source)
	      root.source = root.source->next;
	  else
	      output_error("'NEXT' applied to something not an individual, source, or index\n");
      }
    } else {
	output_error("Newline expected");
    }
  } else if(!strcmp(buf, "PEDIGREE")) {
    variable(ofile);
    if(*template == '\n') {
      template++;
      if(!skipping) {
	  if(current_type == T_INDIV) {
	      if(pedigree_depth)
		  output_pedigree(ofile, current_value.indiv, pedigree_depth);
	  } else {
	      output_error("Attempt to output pedigree for non-individual\n");
	  }
      }
    } else {
	output_error("Newline expected");
    }
  } else if(!strcmp(buf, "INCLUDE")) {
    char path[FILENAME_MAX+1], *pp;
    FILE *incf;

    skip_white_space();
    for(pp = path; pp - path <= FILENAME_MAX
	&& *template && *template != '\n'; ) {
      if(*template == '@') {
	template++;
	if(*template == '@') {
	  template++;
	  *pp++ = '@';
	} else {
	  char *id;
	  if(root_type == T_INDIV)
	      id = root.indiv->xref;
	  else
	      output_error("Use of '!INCLUDE @' on non-individual\n");
	  while(*id && pp - path <= FILENAME_MAX)
	    *pp++ = *id++;
	}
      } else {
	*pp++ = *template++;
      }
    }
    *pp = '\0';
    if((incf = fopen(path, "r")) != NULL) {
      char c;
      while((c = fgetc(incf)) != EOF)
	fputc(c, ofile);
      fclose(incf);
    }
  } else if(!strcmp(buf, "DATE")) {
    if(*template == '\n') {
      template++;
      if(!skipping) {
	  time_t tm = time(NULL);
	  fprintf(ofile, "%s", ctime(&tm));
      }
    } else {
	output_error("Newline expected");
    }
  } else if(!strcmp(buf, "VERSION")) {
    if(*template == '\n') {
      template++;
      if(!skipping) {
	  fprintf(ofile, "%s\n", VERSION);
      }
    } else {
	output_error("Newline expected");
    }
  } else {
    output_error("Unrecognized control command");
  }
}

/*
 * Pick up the next identifier in the template.
 */

#define IDENTIFIER_MAX 63

void collect_identifier(char **ret)
{
  static char id[IDENTIFIER_MAX+1];
  char *ip, c;

  ip = id;
  while(((c = *template) >= 'A' && c <= 'Z')
	|| (c >= 'a' && c <= 'z')) {
    template++;
    if(ip - id < IDENTIFIER_MAX)
      *ip++ = c;
  }
  *ip = '\0';
  *ret = id;
}

void skip_white_space()
{
  while(*template == ' ' || *template == '\t')
    template++;
}

struct binding {
  char *name;
  int value;
  struct binding *next;
} *environment;

void set_variable(char *name, int value)
{
  struct binding *b;
  for(b = environment; b != NULL; b = b->next) {
    if(!strcmp(name, b->name)) {
      b->value = value;
      return;
    }
  }
  if((b = malloc(sizeof(struct binding))) == NULL)
    out_of_memory();
  if((b->name = strdup(name)) == NULL)
    out_of_memory();
  b->value = value;
  b->next = environment;
  environment = b;
}

int get_variable(char *name)
{
  struct binding *b;
  for(b = environment; b != NULL; b = b->next) {
    if(!strcmp(name, b->name))
      return(b->value);
  }
  set_variable(name, 0);
  return(0);
}

void output_error(char *msg)
{
  char *tp;
  int line = 1;

  for(tp = template_start; tp < template; tp++)
    if(*tp == '\n')
      line++;
  fprintf(stderr, "Output error: ");
  fprintf(stderr, "%s template line %d: %s\n",
	  template_start == individual_template ? "individual" :
	  template_start == index_template ? "index" : "surname",
	  line, msg);
}

void construct_url(char *dest)
{
  char url[FILENAME_MAX+1];
  char id[9];

  *dest = '\0';
  if(current_type == T_INDIV) {
      if(indivs_per_directory) {
	  if(!doing_index)
	      sprintf(dest, "../");
	  sprintf(url, "D%04d/",
		  current_value.indiv ?
		  ((current_value.indiv->serial - 1) / indivs_per_directory) + 1 :
		  0);
      } else {
	  sprintf(url, "%s", "");
      }
      strcat(dest, url);
      if(indivs_per_file) {
	  sprintf(id, "G%07d",
		  current_value.indiv ?
		  ((current_value.indiv->serial - 1) / indivs_per_file) + 1: 
		  0);
	  sprintf(url, file_template, id);
	  strcat(dest, url);
	  sprintf(url, "#%s",
		  current_value.indiv ? current_value.indiv->xref : "");
	  strcat(dest, url);
      } else {
	  sprintf(url, file_template,
		  current_value.indiv ? current_value.indiv->xref : "");
	  strcat(dest, url);
      }
  } else if(current_type == T_INDEX) {
      /*
       * I build the indexes myself, so I assume that current_value.index
       * is nonnull if we arrive here.
       */
      if(current_value.index->id == 0)
	  sprintf(dest, file_template, "PERSONS");
      else {
	  sprintf(id, "IND%04d", current_value.index->id);
	  sprintf(dest, file_template, id);
      }
  } else if(current_type == T_SOURCE) {
      if(indivs_per_directory)
	  sprintf(dest, "../");
      sprintf(url, file_template, "SOURCES");
      strcat(dest, url);
      sprintf(url, "#%s",
	      current_value.source ? current_value.source->xref : "");
      strcat(dest, url);
  } else {
      output_error("Can only make a URL from an individual, index, or source\n");
  }
#ifdef MSDOS
  {
    char *cp;
    for(cp = dest; *cp != '\0' && *cp != '#'; cp++) {
	if(isupper(*cp))
	    *cp = tolower(*cp);
    }
  }
#endif
}
