/*
 * Copyright (c) 1995 Eugene W. Stark
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Eugene W. Stark.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 5. No copying or redistribution in any form for commercial purposes is
 *    permitted without specific prior written permission.
 * 6. This software may be used, for evaluation purposes only, for a period
 *    of no more than fourteen days.  Use of the software for a period beyond
 *    fourteen days, or use of the software for anything other than evaluation
 *    purposes, requires written permission from the author.  Such permission
 *    may be obtained by paying to the author a registration fee,
 *    as described in the documentation accompanying this software.
 *
 * THIS SOFTWARE IS PROVIDED BY EUGENE W. STARK (THE AUTHOR) ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * GEDCOM lineage-linked database structure definitions
 */

/*
 * Record types
 */

typedef enum {
  T_NOTYPE, T_INTEGER, T_STRING, T_PLACE, T_NOTE, T_XREF, T_SOURCE,
  T_FAMEVENT, T_EVENT, T_INDIV, T_FAMILY, T_CONT, T_URL, T_INDEX, T_ATTRIBUTE
} record_type;

/*
 * Counts of various kinds of top-level records
 */

extern int total_individuals;
extern int total_families;
extern int total_events;
extern int total_sources;
extern int total_notes;
extern int total_repositories;
extern int total_submitters;

/*
 * Flag controlling capitalization of surnames
 */

extern int capitalization;

/*
 * Flag controlling strict CONT/CONC distinction regarding newlines
 */

extern int strictlinebreaks;

/*
 * Arrays for each access to top-level records
 */

struct individual_record **all_individuals;
struct family_record **all_familes;
struct source_record *all_sources, *last_source;

/*
 * Access to the hierarchical index tree
 */

extern struct index_node *index_root;
extern struct index_node *surname_head;

/*
 * Parameters controlling production of index
 */

extern int index_depth;
extern int index_width;

/*
 * Database structure definitions
 */

struct individual_record {
  int serial;
  char *xref;
  struct name_structure *personal_name;
  char *title;
  char sex;
  struct attribute *attrs, *lastattr;
  struct xref *fams, *lastfams;
  struct xref *famc, *lastfamc;
  struct xref *sources, *lastsource;
  struct note_structure *notes, *lastnote;
  struct event_structure *events, *lastevent;
  struct individual_record *next;
};

struct family_record {
  char *xref;
  char *refn;
  struct xref *husband;
  struct xref *wife;
  struct xref *children, *lastchild;
  struct xref *sources, *lastsource;
  struct note_structure *notes, *lastnote;
  struct event_structure *events, *lastevent;
  struct family_record *next;
};

struct source_record {
  char *xref;
  struct source_structure *source;
  struct source_record *next;
};

struct event_record {
  char *xref;
  struct event_structure *event;
};

struct note_record {
  char *xref;
  struct note_structure *note;
};

struct source_structure {
  char *text;
  struct continuation *cont;
  struct note_structure *notes, *lastnote;
  struct xref *chain;
};

struct name_structure {
  char *name;
  int surname_start;
  int surname_end;
  char *surname;
};

struct place_structure {
  char *name;
};

struct note_structure {
   char *xref;
   char *text;
   struct continuation *cont;
   struct xref *sources, *lastsource;
   struct note_structure *next;
};

struct event_structure {
  struct tag *tag;
  char *date;
  char *text;
  struct place_structure *place;
  struct xref *sources, *lastsource;
  struct note_structure *notes, *lastnote;
  struct event_structure *next;
};

struct xref {
  char *id;
  record_type type;
  union {
    void *generic;	/* A kludge to enable "oblivious" backpatching */
    struct individual_record *individual;
    struct family_record *family;
    struct source_record *source;
  } pointer;
  struct xref *bpl;	/* Backpatch list */
  struct xref *next;
};

struct continuation {
  char *text;
  int linebreak;			/* Break line before text? */
  struct continuation *next;
};

struct attribute {
  struct tag *tag;
  char *text;
  struct attribute *next;
};

struct index_node {
    int id;				/* UID for constructing file names */
    int level;				/* Level number for output */
    struct individual_record *first;	/* First individual under this index */
    struct individual_record *last;	/* Last individual under this index */
    struct index_node *parent;		/* Parent index node (if any) */
    struct index_node *children;	/* Sub-indexes (if any) */
    struct index_node *next;		/* Next index at same level */
    struct index_node *prev;		/* Previous index at same level */
};

/*
 * Function prototypes
 */

void process_records(struct node *np);
void process_individual_record(struct node *np);
void process_family_record(struct node *np);
struct source_record *process_source_record(struct node *np);
void process_event_record(struct node *np);
void process_note_record(struct node *np);
void process_submitter_record(struct node *np);
void process_repository_record(struct node *np);
struct source_structure *process_source(struct node *np);
struct event_structure *process_event(struct node *np);
struct note_structure *process_note(struct node *np);
struct name_structure *process_name(struct node *np);
struct xref *process_xref(struct node *np);
struct xref *dummy_xref();
void link_records(struct node *np);
void index_individuals();
void index_surnames();
int compare_name(struct individual_record **ipp1,
		 struct individual_record **ipp2);
